/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse AUTO_MOUSE

/////////////////////////////////////////////////////////////////////////////////

// Simple "Automatic Mouse". Simulates scanning the mouse over the full range of
// the screen with the X and Y scanning frequencies being different. TheEmu.

#define MOUSE_SPEED vec2(vec2(0.5,0.577777) * 0.25)
#define MOUSE_POS   vec2((1.0+cos(iGlobalTime*MOUSE_SPEED))*u_WindowSize/2.0)
#define MOUSE_PRESS vec2(0.0,0.0)
#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )

/////////////////////////////////////////////////////////////////////////////////

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;
uniform sampler2D iChannel1;

/////////////////////////////////////////////////////////////////////////////////

/* 
---------------------------------------------------------------------------------------

 "Dawn of the Tentacle" by Pablo Roman Andrioli (Kali)
 
 Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

--------------------------------------------------------------------------------------- 
*/


//smooth version by iq
#define smoothversion 

//remove if you see a flat shaded tentacle 
//(it happens to me in firefox)
#define underwater 


// Raymarching 
const float zoom=.95; 
const float detail=.017;    // distance to stop tracing
const float maxdist=14.;    // scene depth
const int maxsteps=55;      // max ray steps

// Light 
const vec3 lightdir=vec3(.6,-1.,0.); 
const float diffuse=.7;
const float specular=.7;
const float specularexp=5.;

// Tentacle Shape
const float maxiter=28.;    // Max iterations 
const float width=.9; 
const float height=0.52;
const float scaling=0.92;	   // Scaling factor at each iteration
const vec3 rotvector=vec3(1.,0.,0.); // Base vector for rotation
const float anglevary=.2;    // Angle variation at each iteration

// Animation 
const float amplitude=0.55;   
const float speed=0.6;
const float dynscaling=0.9; // Scaling factor for the dynamic fractal animation


//-------------------------------------------------------------------------------------------

vec3 dir;
vec2 pix;
vec2 coord;


// Rotation function included in MathUtils.frag of Syntopia's Fragmentarium 
mat3 rotationMat(vec3 v, float angle)
{
	float c = cos(angle);
	float s = sin(angle);
	
	return mat3(c + (1.0 - c) * v.x * v.x, (1.0 - c) * v.x * v.y - s * v.z, (1.0 - c) * v.x * v.z + s * v.y,
		(1.0 - c) * v.x * v.y + s * v.z, c + (1.0 - c) * v.y * v.y, (1.0 - c) * v.y * v.z - s * v.x,
		(1.0 - c) * v.x * v.z - s * v.y, (1.0 - c) * v.y * v.z + s * v.x, c + (1.0 - c) * v.z * v.z
		);
}


// Water
float water(vec3 p){
	//p.z-=15.;
	float l=length(p.xz);
	p.x+=l*3.;
	p.x-=iGlobalTime*1.5;
	float h=(sin(p.x)+cos(p.z))*.05;
	p.xz*=vec2(3,2.5);
	p.xz+vec2(1.,2.5);
	h+=(cos(p.x)+sin(p.z))*.05;
	h+=texture2D(iChannel0,p.xz*.2).z*max(0.1,.2-l*l*.01); //texture displacement
	return p.y-h;
}


float smin( float a, float b )
{
float k = 0.1; float h = clamp( 0.5 + 0.5*(b-a)/k, 0.0, 1.0 );
return mix( b, a, h ) - k*h*(1.0-h);
}


// Scene DE
vec2 DE(vec3 pos){

	float angle=0.;
	vec3 p=pos;
	int n = 0;
	float sc=1.;
	float time=iGlobalTime*speed+100.;
	float amp=sin(time/2.)*amplitude;
	float dtent=1e10;
	float dd=1e10;
	float minit=maxiter;
	vec3 minp=pos;
	for (float n=0.; n < maxiter; n++) {
		float d1=length(p)-width*sc; 
		float d2=length(p+vec3(width*sc*.75,0.,0.))-width*sc*.4; 
		float d=min(d1,d2);
		if (n<1.) dtent=d;
		if (d<=dtent) { //test min distance and save iteration & vector
			dtent=d;
			minit=n;
			minp=p;
		}
		#ifdef smoothversion
		dd=smin(dd,d);
		#endif
		
		p=p*rotationMat(normalize(rotvector+vec3(0.,sin(iGlobalTime)*.7,0.)),sin(time)*amp+angle);
		p.y-=height*sc; // go up
		sc*=scaling; // scale size
	//	amp*=dynscaling; // scale amplitude
		time/=dynscaling; // scale time
		angle+=radians(anglevary); // vary rotation
	}
		#ifdef smoothversion
		dtent=dd;
		#endif	
	
	dtent+=length(texture2D(iChannel1,vec2(minp.y*2.,atan(minp.x,minp.z)*.8)).xyz)*.025;
	
	float dwat=water(pos);
	float de=min(dwat,dtent);
	float col=0.;
	if (de!=dwat) col=minit+1.; // return coloring parameter
	return vec2(de,col);
}


// finite difference normal
vec3 normal(vec3 pos) {
	vec3 e = vec3(0.0,detail,0.0);
	
	return normalize(vec3(
			DE(pos+e.yxx).x-DE(pos-e.yxx).x,
			DE(pos+e.xyx).x-DE(pos-e.xyx).x,
			DE(pos+e.xxy).x-DE(pos-e.xxy).x
			)
		);	
}

// coloring
vec3 color(float obj, vec3 p) {

	if (obj<1.) { //water
		vec3 tex=texture2D(iChannel0,p.xz*.1).xyz;
		return (tex+.2)*vec3(1.7,1.2,1.2);
	} 
	else if (obj>0.) {//tentacle
		float g=(maxiter-obj)/maxiter;
		return vec3(.9,g+.2,0.6);
	}
	
}

//lighting
vec3 light(vec3 p) {
vec3 ldir=normalize(lightdir);
vec3 n=normal(p);
float diff=max(0.0,dot(-n, ldir))*diffuse;
vec3 r = reflect(ldir,n);
float spec=max(0.,dot(dir,-r));
return vec3(1.,.8,.5)*(diff+pow(spec,specularexp)*specular);	
}

//raymarching
vec3 raymarch(vec3 from, vec3 dir) {
	vec3 p,p2;
	float totdist=0., totdist2=0.;
	vec3 col;
	vec2 d=vec2(0.);
	float d2=100000.;
	//march
	for (int i=1; i<maxsteps; i++) {
		p=from+totdist*dir;
		d=DE(p);
		if (d.x<detail || totdist>maxdist) break;
		#ifdef smoothversion
			totdist+=max(.1,d.x); 
		#else
			totdist+=max(.01,d.x); 
		#endif
	}

	//underwater
	if (d.x<detail && d.y<.5) {
		totdist2=totdist+detail*3.;
		for (int i=1; i<maxsteps; i++) {
			p2=from+totdist2*dir;
			d2=length(p2.xz)-width/scaling+p2.y*.2;
			if (d2<detail || totdist2>maxdist) break;
			totdist2+=d2; 
		}
	}
	
	
	vec3 back=mix(vec3(1.3,.95,.7),vec3(.52,.5,.56) //background gradient
	*1.25,clamp((dir.y+.13)*2.5,0.,1.35)); 
	back+=vec3(1.,1.,.5)*(.04+dot(lightdir,-dir)*.4); //dawn
	
	if (d.x<detail) {
		col=color(d.y,p)*light(p-detail*dir*.5); //apply color+light
		col=mix(col,back,smoothstep(0.,2.2,totdist*.17)); //fog
	} else { //background-sky 
		col=back+.85*vec3(1.,.85,.7)*texture2D(iChannel1,
		vec2(dir.y*4.+.5,atan(dir.z,dir.x)*1.8)*.15+vec2(iGlobalTime*.002)).b*(dir.y*.75+.15)
		*vec3(max(0.1,.5-coord.y*2.5))*1.4;
		col-=vec3(max(0.,coord.y*.3));
	}
	#ifdef underwater
	if (d2<detail) {
		vec3 col2=color(1.1,p2)*light(p+p2-detail*dir*.5); 
		col=mix(col,col2,.3); //underwater
	}
	#endif
	
	col+=vec3(1.,.7,.5)*max(0.,9.-totdist)*.04; //camlight
	return col;
}

void main(void)
{
	//Camera
	pix=gl_FragCoord.xy / iResolution.xy;
	float viewangle=-135.+(iMouse.x/iResolution.x)*100.; 
	mat3 rotview=rotationMat(vec3(0.,1.,0.),radians(viewangle));
	coord = pix-vec2(.5);
	coord.y*=iResolution.y/iResolution.x;
	vec3 from=vec3(0.,2.85,8.)*rotview*zoom;
	dir=normalize(vec3(coord*1.8,-1.))*rotview;
	
	//Draw scene
	vec3 col=vec3(0.);
	col=raymarch(from,dir); 
	
	//adjust levels
	col*=pow(length(col),1.8)*.41; 
	col+=vec3(0.02,-.01,0.05);
	col=mix(col,vec3(length(col))*.6,.3);
	
	gl_FragColor = vec4(col,1);
}